<?php
namespace MascotCoreErecoElementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Elementor Hello World
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TM_Elementor_Floating_Objects extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'tm-ele-floating-objects';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Floating Objects', 'mascot-core-elementor' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'tm-elementor-widget-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'tm' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'mascot-core-hellojs' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'general',
			[
				'label' => esc_html__( 'General', 'mascot-core-elementor' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'custom_css_class',
			[
				'label' => esc_html__( "Custom CSS class", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
			]
		);
		$this->add_control(
			'visible_mobile',
			[
				'label' => esc_html__( "Visible on Mobile Devices?", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);



		
		
		$repeater = new \Elementor\Repeater();
		//image
		$repeater->add_control(
			'image', [
				'label' => esc_html__( "Floating Image", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::MEDIA,
			]
		);


		$repeater->add_control(
			'animation_type', [
				'label' => esc_html__( "Floating Animation Type", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => mascot_core_ereco_elementor_get_animation_type(),
				'default' => 'tm-animation-floating'
			]
		);

		$repeater->add_responsive_control(
			'pos_top',
			[
				'label' => esc_html__( "Top (px or %)", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}}' => 'top: {{VALUE}};bottom:auto;',
				],
			]
		);
		$repeater->add_responsive_control(
			'pos_right',
			[
				'label' => esc_html__( "Right (px or %)", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}}' => 'right: {{VALUE}};left:auto;',
				],
			]
		);
		$repeater->add_responsive_control(
			'pos_bottom',
			[
				'label' => esc_html__( "Bottom (px or %)", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}}' => 'bottom: {{VALUE}};top:auto;',
				],
			]
		);
		$repeater->add_responsive_control(
			'pos_left',
			[
				'label' => esc_html__( "Left (px or %)", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}}' => 'left: {{VALUE}};right:auto;',
				],
			]
		);
		$repeater->add_control(
			'width',
			[
				'label' => esc_html__( "Width (px or %)", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
			]
		);
		$repeater->add_control(
			'height',
			[
				'label' => esc_html__( "Height (px or %)", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
			]
		);
		$repeater->add_control(
			'z_index',
			[
				'label' => esc_html__( "Z Index", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
			]
		);

		$repeater->add_control(
			'opacity',
			[
				'label' => esc_html__( 'Opacity', 'mascot-core-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 50,
				],
			]
		);
		$repeater->add_control(
			'image_custom_css_class',
			[
				'label' => esc_html__( "Image Custom CSS class", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
			]
		);
		$repeater->add_control(
			'image_inline_style',
			[
				'label' => esc_html__( "Image Custom Inline CSS", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				"description" => esc_html__( "Example: top: 12px; left: 100px;", 'mascot-core-elementor' ),
			]
		);


		$this->add_control(
			'floating_objects_array',
			[
				'label' => esc_html__( "Floating Objects", 'mascot-core-elementor' ),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$html = '';
		//classes
		$classes = array();
		$classes[] = 'tm-ele-floating-objects';
		$classes[] = $settings['custom_css_class'];
		if ( $settings['visible_mobile'] != 'yes' ) {
			$classes[] = 'd-none d-lg-block';
		}
		$settings['classes'] = $classes;
	?>
		<div class="<?php echo esc_attr(implode(' ', $classes)); ?>">
	<?php
		if ( $settings['floating_objects_array'] ) { 
			$settings['iter'] = 1;
			foreach (  $settings['floating_objects_array'] as $item ) {
				$item['wrapper_inline_css'] = $this->inline_css( $item );
				$iter = $settings['iter']++;

				$img_classes = array();
				$img_classes[] = 'each-object elementor-repeater-item-' . $item['_id'];
				$img_classes[] = $item['animation_type'];
				$img_classes[] = $item['image_custom_css_class'];
				$item['img_classes'] = $img_classes;
				//Produce HTML version by using the parameters (filename, variation, folder name, parameters, shortcode_ob_start)
				$html .= mascot_core_ereco_elementor_get_shortcode_template_part( 'floating-objects', null, 'floating-objects/tpl', $item, true );
			}
		} 
		echo $html;
	?>
		</div>
	<?php
	}

	/**
	 * Get Wrapper Styles
	 */
	protected function inline_css( $params ) {
		$css_array = array();

		if( $params['image'] != '' ) {
			$image = wp_get_attachment_image_src( $params['image']['id'], 'full');
			$css_array[] = 'background-image: url('.$image[0].')';
		}
		if( $params['width'] != '' ) {
			$css_array[] = 'width: '.$params['width'];
		}
		if( $params['height'] != '' ) {
			$css_array[] = 'height: '.$params['height'];
		}
		if( !empty($params['z_index']) ) {
			$css_array[] = 'z-index: '.$params['z_index'];
		}
		
		if( $params['opacity'] != '' ) {
			$css_array[] = 'opacity: '.$params['opacity']['size']*0.01;
		}

		$css_array = implode( '; ', $css_array ).';';

		if( $params['image_inline_style'] != '' ) {
			$css_array .= $params['image_inline_style'];
		}
		return $css_array;
	}
}
